/***************************************************************************
 *
 * Copyright (C) 2001 International Business Machines
 * All rights reserved.
 *
 * This file is part of the GPFS mmfslinux kernel module.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer. 
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written
 *     permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *************************************************************************** */
/* Program to dump kernel memory by address or symbol on a running system.
   Can also get formatted dumps of some kernel data structures.  Invoke
   as 'kdump /var/mmfs/tmp/complete.map.'. */

/*
 * Contents:
 *   Init
 *   CopyString
 *   SymbolHash
 *   LookupSymbolByName
 *   AddUniqueSymbol
 *   AddSymbol
 *   SymbolAddrCompare
 *   SortSymbols
 *   LookupSymbolByAddr
 *   ExactAddrToSymbol
 *   ReadKernelMap
 *   ReadKernel
 *   GetSymbolValue
 *   kMalloc
 *   kFree
 *   CondFree
 *   DumpMemory
 *   GetHex
 *   PrintCxiNode_t
 *   Usage
 *   DoHelp
 *   main
 *
 * $Id: kdump.c,v 1.7.2.1 2002/05/21 21:44:58 dcraft Exp $
 *
 * $Log: kdump.c,v $
 * Revision 1.7.2.1  2002/05/21 21:44:58  dcraft
 * Pull GPFS 1.2.1 up to kernel 2.4.18.
 * mmfsfuncs.Linux must be distributed with /usr/lpp/mmfs/src
 * on developerworks.
 *
 * Revision 1.8  2002/01/22 22:19:57  wyllie
 * Make kdump work for 2.4.2-2 kernel; add more trace code
 *
 * Revision 1.7  2001/09/28 20:45:48  wyllie
 * Add kdump commands to give formatted dumps of the Linux task_struct and
 * various virtual memory structures.
 *
 * Revision 1.6  2001/09/20 05:57:16  wsawdon
 * Renamed va_xperm to va_xinfo and definef bit flags to allow var to be
 * shared between extended permissions and snapLatest entries.
 *
 * Revision 1.5  2001/07/19 23:25:00  dcraft
 * Modified linux trace to allow non blocking trace record
 * writes (format is TRACE?N).  New gpfs swapd process created
 * which is responsible for reclaiming inodes (5 percent every
 * time it runs).  Marked all our inodes so that they would be
 * ignored by linux kswapd.  Added "unused" inode to inode
 * cache that could be used as a signal that linux kswapd is
 * running and kick off gpfs swapd.  Added means to ignore attempts
 * to kill mmfsd by kswapd if the system gets low on memory.
 * All done in an attempt to avoid kswapd premature wakeup on simple
 * locks and mutexes.
 *
 * Revision 1.4  2001/05/30 20:41:58  wyllie
 * Purge Linux inode when give up token for a file in the stat cache.
 *
 * Revision 1.3  2001/05/09 00:43:33  wsawdon
 * Fixed bugs in linux readdir returning internal inode number
 * instead of external snapshot/inode numbers.
 *
 * Revision 1.2  2001/05/01 20:06:53  wsawdon
 * Fix for Raleigh defect 1947.
 * Linux kernel code must locate gnodes in the same
 * way that it locates its inode structures.
 *
 * Revision 1.1  2001/03/16 21:07:34  wyllie
 * Program to dump kernel memory by address or symbol on a running system.
 * Can also get formatted dumps of some kernel data structures.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <termios.h>

#include <cxiTypes.h>

#include <kdump.h>
#include <kdump-kern.h>

/* String storage area.  Strings are allocated at successively higher
   addresses within this area, and are never deallocated. */
char* StringAreaP = NULL;
#define STRING_AREA_SIZE (4*1024*1024)
int StringAreaSizeLeft;


/* Symbol storage area.  Each symbol has an address, some flags, and
   a name.  Symbols are linked into a hash table by name.  Although the
   symbol array itself is not sorted, the auxiliary array SymsByAddrPP is
   sorted by address to allow fast binary search by address. */
int NSymbols = 0;
#define MAX_SYMBOLS (128*1024)
struct Symbol* SymbolAreaP = NULL;
int SymbolTableSorted = 0;
struct Symbol** SymsByAddrPP = NULL;
#define SYM_HASH_BUCKETS 4093
struct Symbol* SymbolHashAnchor[SYM_HASH_BUCKETS];

/* Handle of /dev/kmem */
int KmemHandle = -1;

/* Bounds of the initial portion of kernel virtual memory.  These are
   initialized to liberal values, then tightened by reading kernel
   variables. */
unsigned long LowKernelAddr = 4096;
unsigned long HighKernelAddr = -1L;

/* Head of list of vm_struct structs copied from the kernel.  These define
   which areas of kernel virtual memory are legal to access through
   /dev/kmem. */
struct vmStruct* vmListHeadP = NULL;


/* Initialize */
void Init()
{
  int i;
  struct termios term;
  int rc;

  /* Initialize string area */
  StringAreaP = (char*) malloc(STRING_AREA_SIZE);
  assert(StringAreaP != NULL);
  StringAreaSizeLeft = STRING_AREA_SIZE;

  /* Allocate storage for symbols and put one symbol in the table at
     address 0 */
  SymbolAreaP = (struct Symbol*) malloc(MAX_SYMBOLS*sizeof(struct Symbol));
  assert(SymbolAreaP != NULL);
  for (i=0 ; i<SYM_HASH_BUCKETS ; i++)
    SymbolHashAnchor[i] = NULL;
  AddUniqueSymbol(0, 0, "NULL");

  /* Open /dev/kmem */
  KmemHandle = open("/dev/kmem", O_RDONLY, 0);
  if (KmemHandle == -1)
  {
    fprintf(stderr, "Cannot open /dev/kmem\n");
    perror("open");
    exit(1);
  }
}


/* Copy a string into the string storage area and return a pointer to
   the copy of the string */
static char* CopyString(const char* p)
{
  char* newStrP = StringAreaP;
  int len = strlen(p);
  assert(len < StringAreaSizeLeft);
  strcpy(newStrP, p);
  StringAreaP += len+1;
  StringAreaSizeLeft -= len+1;
  return newStrP;
}


/* Hash function for symbol table names */
static int SymbolHash(const char* nameP)
{
  unsigned int hash = 0;
  int i;
  int len = strlen(nameP);
  for (i=0 ; i<len ; i++)
    hash = (hash << 4) ^ nameP[i];
  return hash % SYM_HASH_BUCKETS;
}


/* Look up symbol by name.  Returns a pointer to the symbol table entry
   if a matching name is found, otherwise NULL. */
struct Symbol* LookupSymbolByName(const char* nameP)
{
  struct Symbol* p;
  int h = SymbolHash(nameP);
  p = SymbolHashAnchor[h];
  while (p != NULL)
  {
    if (strcmp(p->nameP, nameP) == 0)
    {
      DBG(printf("LookupSymbolByName: found '%s' addr 0x%lX\n", nameP, p->addr));
      return p;
    }
    p = p->hashNextP;
  }
  return NULL;
}


/* Add a symbol to the table.  The symbol name must not already be present.
   Returns the address of the new symbol table entry. */
struct Symbol* AddUniqueSymbol(unsigned long addr, int flags, char* nameP)
{
  struct Symbol* p;
  int h = SymbolHash(nameP);
  p = SymbolHashAnchor[h];
  while (p != NULL)
  {
    if (strcmp(p->nameP, nameP) == 0)
    {
      fprintf(stderr, "Symbol %s already in table with addr 0x%lX\nFailed to add symbol at addr 0x%lX\n",
              nameP, p->addr, addr);
      assert(!"symbol not unique in AddUniqueSymbol");
    }
    p = p->hashNextP;
  }

  assert(NSymbols < MAX_SYMBOLS);
  p = &SymbolAreaP[NSymbols];
  NSymbols += 1;
  p->addr = addr;
  p->flags = flags;
  p->nameP = CopyString(nameP);
  p->hashNextP = SymbolHashAnchor[h];
  SymbolHashAnchor[h] = p;
  SymbolTableSorted = 0;
  return p;
}


/* Add a symbol to the table.  If the symbol name is already present,
   append the address of the symbol to the symbol name to make it
   unique.  Returns the address of the new symbol table entry. */
struct Symbol* AddSymbol(unsigned long addr, int flags, char* nameP)
{
  struct Symbol* p = LookupSymbolByName(nameP);
  char* uniqueNameP;
  int i;

  if (p == NULL)
    return AddUniqueSymbol(addr, flags, nameP);

  uniqueNameP = (char*) malloc(strlen(nameP) + 32);
  assert (uniqueNameP != NULL);
  sprintf(uniqueNameP, "%s.%lX", nameP, addr);
  for (i=1 ; ; i++)
  {
    p = LookupSymbolByName(uniqueNameP);
    if (p == NULL)
      break;
    sprintf(uniqueNameP, "%s.%lX.%d", nameP, addr, i);
  }

  p = AddUniqueSymbol(addr, flags, uniqueNameP);
  free(uniqueNameP);
  return p;
}


/* Comparision routine used by SortSymbols.  Compares the addr fields
   of the Symbol objects reachable through the pointers given as arguments. */
static int SymbolAddrCompare(const void* arg1P, const void* arg2P)
{
  struct Symbol** s1PP = (struct Symbol**) arg1P;
  struct Symbol** s2PP = (struct Symbol**) arg2P;
  unsigned long addr1 = (*s1PP)->addr;
  unsigned long addr2 = (*s2PP)->addr;
  if (addr1 < addr2) return -1;
  if (addr1 > addr2) return 1;
  return 0;
}


/* Build an up-to-date copy of the SymsByAddrPP array and sort it in order
   by the addr fields of the associated symbols */
static void SortSymbols()
{
  int i;

  if (SymbolTableSorted)
    return;

  if (SymsByAddrPP != NULL)
    free(SymsByAddrPP);

  SymsByAddrPP = (struct Symbol**) malloc(NSymbols * sizeof(struct Symbol**));
  assert(SymsByAddrPP != NULL);
  for (i=0 ; i<NSymbols ; i++)
    SymsByAddrPP[i] = &SymbolAreaP[i];

  qsort((void*)SymsByAddrPP, NSymbols, sizeof(struct Symbol*), SymbolAddrCompare);
  SymbolTableSorted = 1;
}


/* Look up symbol by address.  Returns a pointer to the symbol table entry
   with the largest address that is less than or equal to the given
   address.  There will always be a symbol in the table with address 0,
   so this can always return a pointer to some Symbol object. */
struct Symbol* LookupSymbolByAddr(unsigned long addr)
{
  int low, high, mid;
  unsigned long addrMid;

  SortSymbols();

  /* Binary search */
  low = 0;
  high = NSymbols - 1;
  while (low < high)
  {
    mid = (low+high+1) / 2;
    addrMid = SymsByAddrPP[mid]->addr;
    if (addrMid > addr)
      high = mid - 1;
    else
      low = mid;
  }
  return SymsByAddrPP[low];
}


/* Look up a symbol by address.  If an exact match for the address is
   found, return a pointer to the symbol name, otherwise return a pointer
   to an empty string. */
const char* ExactAddrToSymbol(unsigned long addr)
{
  static const char* emptyStringP = "";
  struct Symbol* sP;

  if (addr == 0)
    return emptyStringP;
  sP = LookupSymbolByAddr(addr);
  if (sP == NULL)
    return emptyStringP;
  else
    return sP->nameP;
}


/* Read in the map file containing kernel and kernel module symbols */
static void ReadKernelMap(const char* filenameP)
{
  FILE* mapFileP;
  char lineP[4096];
  char* p;
  int len;
  int n;
  unsigned long addr;
  char flagChar;
  int flags;
  char symbolName[4096];

  mapFileP = fopen(filenameP, "r");
  if (mapFileP == NULL)
  {
    fprintf(stderr, "Cannot open kernel map file %s\n", filenameP);
    exit(1);
  }

  for(;;)
  {
    p = fgets(lineP, sizeof(lineP), mapFileP);
    if (p != lineP)
      break;
    len = strlen(lineP);
    if (lineP[len-1] != '\n')
    {
      fprintf(stderr, "Line overflow reading map file: '%s'\n",
              lineP);
      exit(1);
    }
/*     lineP[len-1] = '\0'; */
    /* Examples of input lines:
       c0100000 A _text
       c0314030 B vmlist
       c88cc364 T lockStateToString__5LkObjUxPc
     */
    n = sscanf(lineP, "%lX %c %s\n", &addr, &flagChar, symbolName);
    if (n != 3)
    {
      fprintf(stderr, "Parse error in map file: '%s'\n", lineP);
      exit(1);
    }
    DBG(printf("map line: 0x%lX flag '%c' symbol '%s'\n",
               addr, flagChar, symbolName));
    switch (flagChar)
    {
      case 't':
      case 'T':
        flags = FL_TEXT;
        break;

      case 'b':
      case 'B':
        flags = FL_BSS;
        break;

      case 'd':
      case 'D':
        flags = FL_DATA;
        break;

      case 'r':
      case 'R':
        flags = FL_RODATA;
        break;

      case 'A':
        flags = FL_SECTION;
        break;

      case '?':
        flags = FL_KSTRTAB;
        break;

      default:
        flags = 0;
    }
    AddSymbol(addr, flags, symbolName);
  }
  fclose(mapFileP);
}


/* Read kernel memory after checking the address for validity.  If the address
   is invalid, return -1, otherwise return 0. */
int ReadKernel(unsigned long addr, void* bufP, int len)
{
  long long desiredOffset;
  long long actualOffset;
  int lenRead;
  struct vmStruct* aListP;

  /* Check address for validity */
  if (addr < LowKernelAddr)
  {
    DBG(printf("ReadKernel: addr 0x%lX < LowKernelAddr 0x%lX\n",
               addr, LowKernelAddr));
    return -1;
  }
  if (addr > HighKernelAddr)
  {
    aListP = vmListHeadP;
    while (aListP != NULL)
    {
      if (addr >= aListP->startAddr  &&
          addr+len < aListP->startAddr+aListP->areaLen)
      {
        DBG(printf("ReadKernel: addr 0x%lX in range starting at 0x%lX\n",
                   addr, aListP->startAddr));
        goto addrOK;
      }
      aListP = aListP->nextAreaP;
    }
    DBG(printf("ReadKernel: addr 0x%lX out of range\n", addr));
    return -1;
  }

addrOK:
  desiredOffset = addr;
  actualOffset = lseek64(KmemHandle, desiredOffset, SEEK_SET);
  if (actualOffset != desiredOffset)
  {
    fprintf(stderr, "offset set to 0x%llX instead of 0x%llX\n",
            actualOffset, desiredOffset);
    perror("llseek");
    exit(1);
  }
  lenRead = read(KmemHandle, bufP, len);
  if (lenRead != len)
  {
    fprintf(stderr, "%d bytes read at addr 0x%lX\n", lenRead, addr);
    if (lenRead == -1)
      perror("read");
    return -1;
  }
  return 0;
}


/* Read the current value of a kernel symbol.  Returns 0 on success, -1
   on failure. */
int GetSymbolValue(const char* nameP, void* bufP, int len)
{
  int rc;
  struct Symbol* sP;
  sP = LookupSymbolByName(nameP);
  if (sP == NULL)
    return -1;
  rc = ReadKernel(sP->addr, bufP, len);
  DBG(if (len == sizeof(long))
        printf("GetSymbolValue: symbol %s has value 0x%X\n",
               nameP, *((long*)bufP)));
  return rc;
}


/* Wrapper around malloc so it can be called from modules that include
   kernel files */
void* kMalloc(int len)
{
  return malloc(len);
}


/* Wrapper around free so it can be called from modules that include
   kernel files */
void kFree(void* p)
{
  free(p);
}


/* Conditional free.  Free storage if p is not NULL. */
void CondFree(void* p)
{
  if (p != NULL)
    free(p);
}


/* Dump a block of memory.  Duplicate lines are supressed. */
void DumpMemory(char* p, int nBytes, unsigned long origin, int dumpFlags)
{
  char* origP;
  union
  {
    unsigned char i1[16];
    unsigned short i2[8];
    unsigned int i4[4];
    unsigned long long int i8[2];
  } buf;
  int row, col, lineLen, oLen;
  char ch;
  Boolean showedDots;
  int i;
  char line[128];

  origP = p;
  if (dumpFlags & DUMPMEM_ROUND)
  {
    p -= origin & 0x00000003;
    nBytes += origP - p;
    origin -= origP - p;
  }
  nBytes = 16 * ((nBytes+15) / 16);

  buf.i1[0] = *p + 1;

  showedDots = false;
  for (row=0 ; row<nBytes/16 ; row++)
  {
    if (memcmp(p, &buf, 16) != 0)
    {
      memcpy(&buf, p, 16);

      /* Dump label of this line.  Save length to indent later lines. */
      lineLen = sprintf(line, "%08x: ", origin + row*16);
      oLen = lineLen;

      /* Dump 16 bytes in hex, in address order */
      lineLen += sprintf(line+lineLen,
                         "%02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x  *",
                         buf.i1[0], buf.i1[1], buf.i1[2], buf.i1[3],
                         buf.i1[4], buf.i1[5], buf.i1[6], buf.i1[7],
                         buf.i1[8], buf.i1[9], buf.i1[10], buf.i1[11],
                         buf.i1[12], buf.i1[13], buf.i1[14], buf.i1[15]);

      /* Dump 16 bytes as chars, translating unprintable chars to . */
      for (col=0 ; col<16 ; col++)
      {
        ch = buf.i1[col];
        if (!isalnum (ch)  &&
            !ispunct (ch))
          line[lineLen] = '.';
        else
        {
          line[lineLen] = ch;
          if (ch == '%')
          {
            lineLen += 1;
            line[lineLen] = '%';
          }
        }
        lineLen += 1;
      }
      sprintf(line+lineLen, "*\n");
      printf(line);

      /* If requested, dump 16 bytes as 8 shorts */
      if (dumpFlags & DUMPMEM_I2)
      {
        for (i=0 ; i<oLen ; i++)
          line[i] = ' ';
        lineLen = oLen;
        lineLen += sprintf(line+lineLen,
                           "%04x %04x %04x %04x %04x %04x %04x %04x",
                           buf.i2[0], buf.i2[1], buf.i2[2], buf.i2[3],
                           buf.i2[4], buf.i2[5], buf.i2[6], buf.i2[7]);
        sprintf(line+lineLen, "\n");
        printf(line);
      }

      /* If requested, dump 16 bytes as 4 ints */
      if (dumpFlags & DUMPMEM_I4)
      {
        for (i=0 ; i<oLen ; i++)
          line[i] = ' ';
        lineLen = oLen;
        lineLen += sprintf(line+lineLen, "%08x %08x %08x %08x",
                           buf.i4[0], buf.i4[1], buf.i4[2], buf.i4[3]);
        sprintf(line+lineLen, "\n");
        printf(line);
      }

      /* If requested, dump 16 bytes as 2 long long ints */
      if (dumpFlags & DUMPMEM_I8)
      {
        for (i=0 ; i<oLen ; i++)
          line[i] = ' ';
        lineLen = oLen;
        lineLen += sprintf(line+lineLen, "%016llx  %016llx",
                           buf.i8[0], buf.i8[1]);
        sprintf(line+lineLen, "\n");
        printf(line);
      }

      showedDots = false;
    }
    else
    {
      if (!showedDots)
      {
        printf(" ...\n");
        showedDots = true;
      }
    }
    p += 16;
  }

} /* end of DumpMemory */


/* Convert a hex string to a number.  Sets *rcP to 0 if successful, nonzero
   otherwise */
static unsigned long GetHex(const char* argP, int* rcP)
{
  char* p = NULL;
  unsigned long result;

  result = strtoul(argP, &p, 16);
  if (*p != '\0')
    *rcP = 1;
  else
    *rcP = 0;
  DBG(printf("strtoul argP 0x%lX arg '%s' result 0x%lX p 0x%lX *rcP %d\n",
             argP, argP, result, p, *rcP));
  return result;
}


/* Print a GPFS cxiNode_t.  Parameter may be NULL. */
void PrintCxiNode_t(void* parm, unsigned long addr)
{
  cxiNode_t* cP = (cxiNode_t*) parm;

  if (cP == NULL)
  {
    printf("NULL cxiNode_t\n");
    return;
  }

  printf("cxiNode_t 0x%lX:\n", addr);

  printf("  osNodeP         inode 0x%lX\n", cP->osNodeP);
  printf("  nType           %s\n",
         cP->nType == cxiVNON  ? "cxiVNON" :
         cP->nType == cxiVREG  ? "cxiVREG" :
         cP->nType == cxiVDIR  ? "cxiVDIR" :
         cP->nType == cxiVBLK  ? "cxiVBLK" :
         cP->nType == cxiVCHR  ? "cxiVCHR" :
         cP->nType == cxiVLNK  ? "cxiVLNK" :
         cP->nType == cxiVSOCK ? "cxiVSOCK" :
         cP->nType == cxiVBAD  ? "cxiVBAD" :
         cP->nType == cxiVFIFO ? "cxiVFIFO" :
         cP->nType == cxiVMPC  ? "cxiVMPC" :
         "??");
  printf("  mapSeg            0x%lX\n", cP->mapSeg);
  printf("  mapReadCount      %d\n", cP->mapReadCount);
  printf("  mapWriteCount     %d\n", cP->mapWriteCount);
  printf("  readCount         %d\n", cP->readCount);
  printf("  writeCount        %d\n", cP->writeCount);
  printf("  execCount         %d\n", cP->execCount);
  printf("  icValid           0x%X", cP->icValid);
  if (cP->icValid & CXI_IC_PERM) printf(" CXI_IC_PERM");
  if (cP->icValid & CXI_IC_ATTR) printf(" CXI_IC_ATTR");
  printf("\n");
  printf("  xinfo             0x%X\n", cP->xinfo);
  printf("  nfsP              0x%lX\n", cP->nfsP);
  printf("  mmapFlush         %d\n", cP->mmapFlush);
  printf("  destroyIfDelInode %d\n", cP->destroyIfDelInode);
  printf("  invalidateCalled  %d\n", cP->invalidateCalled);
}


static void Usage()
{
  fprintf(stderr, "Usage: kdump map-file\n");
  exit(1);
}


static void DoHelp()
{
  printf("Commands:\n"
"  address_space addr  - dump address_space\n"
"  cxiNode_t addr      - dump cxiNode_t\n"
"  dentry addr         - dump dcache dentry\n"
"  inode addr          - dump inode\n"
"  mm_struct addr      - dump mm_struct\n"
"  page addr           - dump struct page\n"
#ifdef GPFS_ARCH_I386
"  pgd addr vaddr      - translate vaddr using pgd at addr\n"
#endif
"  super_block addr    - dump super_block\n"
"  task pid            - dump task_struct for pid\n"
"  task_struct addr    - dump task_struct\n"
"  vm_area_struct addr - dump vm_area_struct\n"
"  mem addr len        - dump memory at addr\n"
"  mem symbol len      - dump memory at address of symbol\n"
"  q                   - quit\n");
}


main(int argc, char* argvPP[])
{
  struct Symbol* sP;
  char lineP[1024];
  char* p;
  char* cmdP;
  char* arg1P;
  char* arg2P;
  char* arg3P;
  unsigned long addr;
  unsigned long vaddr;
  unsigned long pid;
  int len;
  void* x;
  int rc;

  if (argc < 2)
    Usage();

  Init();
  ReadKernelMap(argvPP[1]);
  KernInit();

  for (;;)
  {
    printf(">");
    p = fgets(lineP, sizeof(lineP), stdin);
    if (p == NULL)
      break;
    p = lineP;
    while (isspace(*p)) p += 1;
    if (*p == '\0')
      continue;
    cmdP = strtok(lineP, " \t\n");
    arg1P = strtok(NULL, " \t\n");
    arg2P = strtok(NULL, " \t\n");
    arg3P = strtok(NULL, " \t\n");

    if (strcmp(cmdP, "?") == 0)
      DoHelp();
    else if (strcmp(cmdP, "q") == 0)
      break;
    else if (strcmp(cmdP, "quit") == 0)
      break;
    else if (strcmp(cmdP, "address_space") == 0)
    {
      addr = strtoul(arg1P, NULL, 16);
      x = GetAddressSpace(addr);
      PrintAddressSpace(x, addr);
      CondFree(x);
    }
    else if (strcmp(cmdP, "cxiNode_t") == 0)
    {
      addr = strtoul(arg1P, NULL, 16);
      x = GenericGet(addr, sizeof(cxiNode_t));
      PrintCxiNode_t(x, addr);
      CondFree(x);
    }
    else if (strcmp(cmdP, "dentry") == 0)
    {
      addr = strtoul(arg1P, NULL, 16);
      x = GetDentry(addr);
      PrintDentry(x, addr);
      FreeDentry(x);
    }
    else if (strcmp(cmdP, "inode") == 0)
    {
      addr = strtoul(arg1P, NULL, 16);
      x = GetInode(addr);
      PrintInode(x, addr);
      CondFree(x);
    }
    else if (strcmp(cmdP, "mem") == 0)
    {
      sP = LookupSymbolByName(arg1P);
      if (sP != NULL)
        addr = sP->addr;
      else
      {
        addr = GetHex(arg1P, &rc);
        if (rc != 0)
        {
          printf("Bad address %s\n", arg1P);
          continue;
        }
      }
      len = strtol(arg2P, NULL, 0);
      x = malloc(len);
      if (x != NULL)
      {
        rc = ReadKernel(addr, x, len);
        if (rc != 0)
          printf("Error reading from 0x%lX for %d bytes\n", addr, len);
        else
          DumpMemory((char*)x, len, addr, DUMPMEM_I4);
        free(x);
      }
    }
    else if (strcmp(cmdP, "mm_struct") == 0)
    {
      addr = strtoul(arg1P, NULL, 16);
      x = GetMMStruct(addr);
      PrintMMStruct(x, addr);
      CondFree(x);
    }
    else if (strcmp(cmdP, "page") == 0)
    {
      addr = strtoul(arg1P, NULL, 16);
      x = GetPage(addr);
      PrintPage(x, addr);
      CondFree(x);
    }
#ifdef GPFS_ARCH_I386
    else if (strcmp(cmdP, "pgd") == 0)
    {
      addr = strtoul(arg1P, NULL, 16);
      vaddr = strtoul(arg2P, NULL, 16);
      VirtToReal(addr, vaddr);
    }
#endif
    else if (strcmp(cmdP, "task") == 0)
    {
      pid = strtoul(arg1P, NULL, 10);
      x = GetTaskStructByPID(pid, &addr);
      if (x != NULL)
      {
        PrintTaskStruct(x, addr);
        CondFree(x);
      }
    }
    else if (strcmp(cmdP, "task_struct") == 0)
    {
      addr = strtoul(arg1P, NULL, 16);
      x = GetTaskStruct(addr);
      PrintTaskStruct(x, addr);
      CondFree(x);
    }
    else if (strcmp(cmdP, "super_block") == 0)
    {
      addr = strtoul(arg1P, NULL, 16);
      x = GetSuperBlock(addr);
      PrintSuperBlock(x, addr);
      CondFree(x);
    }
    else if (strcmp(cmdP, "vm_area_struct") == 0)
    {
      addr = strtoul(arg1P, NULL, 16);
      x = GetVMAreaStruct(addr);
      PrintVMAreaStruct(x, addr);
      CondFree(x);
    }

    else
      printf("Bad command '%s'\n", cmdP);
  }
}

